<?php

// CoreSlim - Última atualização 02/03/2023 15:26

namespace CoreSlim;

use CoreSlim\Validation;

class Treatment
{
    public static function handle($var, $filter = 'sanitize')
    {
        switch($filter)
        {
            case 'injection':
            case 'sanitize':
                return filter_var($var, FILTER_SANITIZE_URL);
            break;

            case 'int':
                return intval($var);
            break;

            case 'string':
                return self::handle($var, 'no-tags');
            break;

            case 'text':
                if ($var) {
                    return trim(addslashes($var));
                }
            break;

            case 'text-clean':
                if ($var) {
                    return self::rteSafe(trim(rtrim(addslashes($var))));
                }
            break;

            case 'date':
                return self::date($var, 'mysql');
            break;

            case 'amount': return number_format(str_replace(',', '.', str_replace('.', '', $var)), 2, '.', ''); break;

            case 'no-tags': if ($var) { return addslashes(strip_tags(str_replace('"', '\'\'', $var))); } break;

            case 'cpf':
                if (Validation::cpf($var)) {
                    return $var;
                }
            break;

            case 'email':
                return strtolower($var);
            break;

            case 'phone':
                return str_replace('_', '', $var);
            break;

            case 'array':
                return implode('-', addslashes(str_replace('"', '\'\'', $var)));
            break;

            case 'array_explode':
                return str_replace('"', '\'\'', $var);
                break;

            case 'array_explode_all':
                $varReturn = null;

                foreach (self::handle($var, 'array_explode') as $key => $value) {
                    if (!empty($value)) {
                        $varReturn[] = str_replace('/', '/', htmlentities(self::handle($value, 'text'), ENT_QUOTES));
                    }
                    else {
                        $varReturn[] = '';
                    }
                }

                return $varReturn;
            break;

            case 'array_explode_shift':
                $varReturn = null;

                array_shift($var);

                foreach (self::handle($var, 'array_explode') as $key => $value) {
                    if (!empty($value)) {
                        $varReturn[] = str_replace('/', '/', htmlentities(self::handle($value, 'text'), ENT_QUOTES));
                    }
                    else {
                        $varReturn[] = '';
                    }
                }

                return $varReturn;
            break;

            case 'json':
                return json_encode(self::handle($var, 'array_explode_all'));
            break;

            case 'json_shift':
                return json_encode(self::handle($var, 'array_explode_shift'));
            break;

            case 'code':
                return base64_encode(htmlentities($var, ENT_QUOTES));
            break;
        }
    }

    public static function string($var, $filter = 'string', $characters = 'all', $tags = false, $encode = 'UTF-8')
    {
        if ($var) {
            $var = htmlspecialchars_decode(self::utfCharacters($var));

            switch($filter)
            {
                case 'string':
                case 'phone':
                case 'email':
                    $var = html_entity_decode($var);

                    if ($characters == 'all' || !$characters) {
                        return stripcslashes($var);
                    }
                    else {
                        if ($tags == false) {
                            $var = strip_tags($var);
                        }

                        if (mb_strlen($var) > $characters) {
                            $var = mb_substr($var, 0, $characters - 2, $encode).'...';
                        }
                        else {
                            $var = mb_substr($var, 0, $characters, $encode);
                        }

                        return stripcslashes($var);
                    }
                break;

                case 'text':
                    return stripcslashes($var);
                break;

                case 'amount':
                    return number_format(stripcslashes($var), 2, ',', '.');
                break;

                case 'amount-twodigits':
                    $amount = explode('.', $var);

                    if (!array_key_exists(1, $amount)) {
                        return $var.'.00';
                    }
                    else {
                        return $var;
                    }
                break;

                case 'amountpoint':
                    return number_format(stripcslashes($var), 2, '.', '.');
                break;

                case 'hour2digits':
                    return substr($var, 0, 5);
                break;

                case 'hour2digitsH':
                    return str_replace(':', 'h', self::date($var, 'hour2digits'));
                break;

                case 'json':
                    return json_decode($var);
                break;

                case 'array':
                    return explode('-', stripcslashes($var));
                break;

                case 'int':
                    if ($var) {
                        $var = str_replace('.', '', $var);
                    }

                    return intval($var);
                break;

                case 'float':
                    return floatval($var);
                break;

                case 'unserializestring':
                    return stripcslashes(html_entity_decode($var, ENT_QUOTES));
                break;

                case 'date':
                    return self::date($var, 'brazil');
                break;

                case 'slug':
                    return self::generateSlug($var);
                break;

                case 'str-pad10':
                    return str_pad($var, 10, '0', STR_PAD_LEFT);
                break;

                case 'cammel-ase':
                    return mb_convert_case(stripcslashes($var), MB_CASE_TITLE, 'UTF-8');
                break;

                case 'callto':
                    return '+55'.preg_replace('/[^0-9]/', '', $var);
                break;

                case 'calltonot+':
                    return '55'.preg_replace('/[^0-9]/', '', $var);
                break;

                case 'cep-slug':
                    return str_replace('-', '', $var);
                break;

                case 'first-name':
                    $explode = explode(' ', $var);

                    return self::string($explode[0]);
                break;

                case 'code':
                    return html_entity_decode(base64_decode($var), ENT_QUOTES);
                break;
            }
        }
    }

    public static function date($date, $filter)
    {
        if ($date && $date != '0000-00-00' && $date != '00:00' && $date != '00:00:00' && $date != '00-00-0000' && $date != '0000-00-00 00:00:00' && $date != '00-00-0000 00:00:00') {
            switch ($filter)
            {
                case 'mysql':
                    return mb_substr($date, 6, 4).'-'.mb_substr($date, 3, 2).'-'.mb_substr($date, 0, 2);
                break;

                case 'day':
                    return mb_substr($date,8,2);
                break;

                case 'month':
                    return mb_substr($date,5,2);
                break;

                case 'year':
                    return mb_substr($date,0,4);
                break;

                case 'hour':
                    return mb_substr($date,11,2).':'.mb_substr($date,14,2).':'.mb_substr($date,17,2);
                break;

                case 'brazil':
                    return self::date($date,'day').'/'.self::date($date,'month').'/'.self::date($date,'year');
                break;

                case 'brazil-daymonth':
                    return self::date($date,'day').'/'.self::date($date,'month');
                break;

                case 'timestamp_brazil':
                    return self::date($date,'day').'/'.self::date($date,'month').'/'.self::date($date,'year').' '.mb_substr($date,11,8);
                break;

                case 'hour_abbr':
                    return mb_substr($date,11,2).'h'.mb_substr($date,14,2);
                break;

                case 'month_name':
                    switch ($date)
                    {
                        case '01': return 'janeiro'; break;
                        case '02': return 'fevereiro'; break;
                        case '03': return 'março'; break;
                        case '04': return 'abril'; break;
                        case '05': return 'maio'; break;
                        case '06': return 'junho'; break;
                        case '07': return 'julho'; break;
                        case '08': return 'agosto'; break;
                        case '09': return 'setembro'; break;
                        case '10': return 'outubro'; break;
                        case '11': return 'novembro'; break;
                        case '12': return 'dezembro'; break;
                    }
                break;

                case 'month_number':
                    switch ($date)
                    {
                        case 'janeiro': return '01'; break;
                        case 'fevereiro': return '02'; break;
                        case 'março': return '03'; break;
                        case 'abril': return '04'; break;
                        case 'maio': return '05'; break;
                        case 'junho': return '06'; break;
                        case 'julho': return '07'; break;
                        case 'agosto': return '08'; break;
                        case 'setembro': return '09'; break;
                        case 'outubro': return '10'; break;
                        case 'novembro': return '11'; break;
                        case 'dezembro': return '12'; break;
                    }
                break;

                case 'month_name_abbr':
                    return substr(self::date(self::date($date, 'month'), 'month_name'), 0, 3);
                break;

                case 'weekday':
                    $weekday = date('w', mktime(0, 0, 0, self::date($date,'month'), self::date($date,'day'), self::date($date,'year')));

                    switch ($weekday)
                    {
                        case '0': return 'Domingo'; break;
                        case '1': return 'Segunda-feira'; break;
                        case '2': return 'Terça-feira'; break;
                        case '3': return 'Quarta-feira'; break;
                        case '4': return 'Quinta-feira'; break;
                        case '5': return 'Sexta-feira'; break;
                        case '6': return 'Sábado'; break;
                    }
                break;

                case 'extensive':
                    return self::date($date,'day').' de '.self::date(self::date($date,'month'),'month_name').' de '.self::date($date,'year');
                break;

                case 'extensive_noyear':
                    return self::date($date,'day').' de '.self::date(self::date($date,'month'),'month_name');
                break;

                case 'extensive_hour':
                    $hour = self::date($date, 'hour_abbr');

                    if ($hour != 'h' && $hour != '00h00') {
                        return self::date($date,'extensive'). ' às '.self::date($date,'hour_abbr');
                    }
                    else {
                        return self::date($date, 'extensive');
                    }
                break;

                case 'extensive_weekday':
                    return self::date($date, 'weekday').', '.self::date($date, 'extensive_hour');
                break;

                case 'extensive_weekday_default':
                    return self::date($date, 'weekday').', '.self::date($date, 'brazil');
                break;

                case 'age':
                    if ($date && $date != '0000-00-00') {
                        $age = floor((time() - strtotime($date)) / (60 * 60 * 24 * 365));

                        return intval($age) . ' anos';
                    }
                break;

                case 'hour2digits':
                    return substr($date, 0, 5);
                break;
            }
        }
    }

    public static function generateSlug($data, $separador = '-')
    {
        $data = trim($data);

        $data = str_replace(array('á', 'ã', 'à', 'ä', 'â', 'ª', 'Á', 'À', 'Â', 'Ã', 'Ä'), array('a', 'a', 'a', 'a', 'a', 'a', 'A', 'A', 'A', 'A', 'A'), $data);
        $data = str_replace(array('é', 'è', 'ë', 'ê', 'É', 'È', 'Ê', 'Ë'), array('e', 'e', 'e', 'e', 'E', 'E', 'E', 'E'), $data );
        $data = str_replace(array('í', 'ì', 'ï', 'î', 'Í', 'Ì', 'Ï', 'Î'), array('i', 'i', 'i', 'i', 'I', 'I', 'I', 'I'), $data);
        $data = str_replace(array('ó', 'õ', 'ò', 'ö', 'ô', 'Ó', 'Ò', 'Õ', 'Ö', 'Ô'), array('o', 'õ', 'o', 'o', 'o', 'O', 'O', 'O', 'O', 'O'), $data);
        $data = str_replace(array('ú', 'ù', 'ü', 'û', 'Ú', 'Ù', 'Û', 'Ü'), array('u', 'u', 'u', 'u', 'U', 'U', 'U', 'U'), $data);
        $data = str_replace(array('ñ', 'Ñ', 'ç', 'Ç'), array('n', 'N', 'c', 'C',), $data);
        $data = str_replace(array("\\", "¨", "º", "~", "#", "@", "|", "!", "\"", "·", "$", "%", "&", "/", "(", ")", "?", "'", "¡", "¿", "[", "^", "`", "]", "+", "}", "{", "¨", "´", ">", "<", ";", ",", ":", "."), '-', $data);
        $data = str_replace("_", "-", $data);
        $data = str_replace(" ", "-", $data);

        $slug = iconv('UTF-8', 'ASCII//TRANSLIT', $data);
        $slug = preg_replace("/[^a-zA-Z0-9\/_|+ -]/", '', $slug);
        $slug = strtolower(trim($slug, $separador));
        $slug = preg_replace("/[\/_|+ -]+/", $separador, $slug);

        return $slug;
    }

    public static function generateSlugOnlyChars($data, $separador = '-')
    {
        $data = trim($data);

        $data = str_replace(['á', 'ã', 'à', 'ä', 'â', 'ª', 'Á', 'À', 'Â', 'Ã', 'Ä'], ['a', 'a', 'a', 'a', 'a', 'a', 'A', 'A', 'A', 'A', 'A'], $data);
        $data = str_replace(['é', 'è', 'ë', 'ê', 'É', 'È', 'Ê', 'Ë'], ['e', 'e', 'e', 'e', 'E', 'E', 'E', 'E'], $data );
        $data = str_replace(['í', 'ì', 'ï', 'î', 'Í', 'Ì', 'Ï', 'Î'], ['i', 'i', 'i', 'i', 'I', 'I', 'I', 'I'], $data);
        $data = str_replace(['ó', 'õ', 'ò', 'ö', 'ô', 'Ó', 'Ò', 'Õ', 'Ö', 'Ô'], ['o', 'õ', 'o', 'o', 'o', 'O', 'O', 'O', 'O', 'O'], $data);
        $data = str_replace(['ú', 'ù', 'ü', 'û', 'Ú', 'Ù', 'Û', 'Ü'], ['u', 'u', 'u', 'u', 'U', 'U', 'U', 'U'], $data);
        $data = str_replace(['ñ', 'Ñ', 'ç', 'Ç'], ['n', 'N', 'c', 'C'], $data);
        $data = str_replace("_", "-", $data);
        $data = str_replace(" ", "-", $data);

        $slug = iconv('UTF-8', 'ASCII//TRANSLIT', $data);
        $slug = mb_strtolower(trim($slug, $separador));

        return $slug;
    }

    public static function utfCharacters($data)
    {
        $charUtf8 = ['u00e1', 'u00e0', 'u00e2', 'u00e3', 'u00e4', 'u00c1', 'u00c0', 'u00c2', 'u00c3', 'u00c4', 'u00e9', 'u00e8', 'u00ea', 'u00ea', 'u00c9', 'u00c8', 'u00ca', 'u00cb', 'u00ed', 'u00ec', 'u00ee', 'u00ef', 'u00cd', 'u00cc', 'u00ce', 'u00cf', 'u00f3', 'u00f2', 'u00f4', 'u00f5', 'u00f6', 'u00d3', 'u00d2', 'u00d4', 'u00d5', 'u00d6', 'u00fa', 'u00f9', 'u00fb', 'u00fc', 'u00da', 'u00d9', 'u00db', 'u00e7', 'u00c7', 'u00f1', 'u00d1', 'u0026', 'u0027'];
        $charIso = ['á', 'à', 'â', 'ã', 'ä', 'Á', 'À', 'Â', 'Ã', 'Ä', 'é', 'è', 'ê', 'ê', 'É', 'È', 'Ê', 'Ë', 'í', 'ì', 'î', 'ï', 'Í', 'Ì', 'Î', 'Ï', 'ó', 'ò', 'ô', 'õ', 'ö', 'Ó', 'Ò', 'Ô', 'Õ', 'Ö', 'ú', 'ù', 'û', 'ü', 'Ú', 'Ù', 'Û', 'ç', 'Ç', 'ñ', 'Ñ', '&', '\''];

        return str_replace($charUtf8, $charIso, $data);
    }

    public static function amount($amount)
    {
        return number_format($amount, 2, '.', '');
    }

    public static function rteSafe($strText)
    {
        $tmpString = $strText;

        $tmpString = str_replace("\'", "'", $tmpString);
        $tmpString = str_replace("'", "\'", $tmpString);
        $tmpString = str_replace("–", "-", $tmpString);
        $tmpString = str_replace('<h1>&nbsp;</h1>', '', $tmpString);
        $tmpString = str_replace('<h2>&nbsp;</h2>', '', $tmpString);
        $tmpString = str_replace('<p>&nbsp;</p>', '', $tmpString);

        return $tmpString;
    }
}
