<?php

// CoreSlim v1.9.7 - Última atualização 08/09/2023 15:07

namespace CoreSlim;

class Upload
{
    private $callback = null;

    private $config = [];

    private $controls = [];

    private $images = [];

    private $files = null;
    private $output = null;

    private $typeFormatCheck = [
        'photo' => [
            'jpg',
            'png',
            'gif',
            'webp'
        ],
        'annex' => [
            'jpg',
            'png',
            'gif',
            'webp',
            'pdf',
            'winzip',
            'winrar',
            'word',
            'excel',
            'powerpoint'
        ],
        'audio' => [
            'audios'
        ]
    ];

    private $mimetypesAcceptFiles = [
        'jpg' => [
            'image/pjpeg',
            'image/jpeg',
            'image/jpg'
        ],
        'png' => [
            'image/png'
        ],
        'gif' => [
            'image/gif'
        ],
        'webp' => [
            'image/webp'
        ],
        'pdf' => [
            'application/pdf'
        ],
        'winzip' => [
            'application/x-zip-compressed',
            'application/zip'
        ],
        'winrar' => [
            'application/octet-stream'
        ],
        'word' => [
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ],
        'excel' => [
            'application/xls',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ],
        'powerpoint' => [
            'application/powerpoint',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation'
        ],
        'txt' => [
            'text/plain'
        ],
        'audios' => [
            'audio/mpeg',
            'audio/x-mpeg',
            'audio/mp3',
            'audio/x-mp3',
            'audio/mpeg3',
            'audio/x-mpeg3',
            'audio/mpg',
            'audio/x-mpg',
            'audio/x-mpegaudio'
        ]
    ];

    public function __construct()
    {
        $this->resetAttrs();
    }

    public function setCallback($callback)
    {
        $this->callback = $callback;
    }

    public function setConfig($type, $value)
    {
        $this->config[$type] = $value;
    }

    public function getConfig($type = null)
    {
        if ($type) {
            $type = explode('.', $type);
        }

        return Tools::getArray($this->config, $type);
    }

    public function setControls($type, $value)
    {
        $this->controls[$type] = $this->controls[$type] + $value;
    }

    public function getControls($type = null)
    {
        if ($type) {
            $type = explode('.', $type);
        }

        return Tools::getArray($this->controls, $type);
    }

    public function setImages($type, $value)
    {
        $this->images[$type] = $value;
    }

    public function getImages($type)
    {
        if ($type) {
            $type = explode('.', $type);
        }

        return Tools::getArray($this->images, $type);
    }

    public function resetAttrs()
    {
        $this->config = [
            'required' => false,
            'field' => null,
            'folder' => null,
            'folderDivider' => false,
            'folderDividerYear' => false,
            'folderDividerMonth' => false,
            'folderDividerDay' => false,
            'route' => null,
            'multiple' => false,
            'name' => 'automatic',

            'type' => 'photo',
            'accept' => 'images',

            'limits' => [
                'itens' => 20,
                'size' => 41943040
            ]
        ];

        $this->controls = [
            'itens' => 0,
            'size' => 0
        ];

        $this->images = [
            'thumb' => true,
            'exact' => false,
            'crop' => true,
            'aspect' => 1,
            'compression' => 100,
            'webpconvert' => true,
            'fallback' => [
                'active' => true,
                'percentage' => 1.2
            ],
            'dimensions' => [
                'default' => [1200, 900],
                'thumb' => [600, 450],
                'min' => [200, 150],
                'max' => [1920, 1080]
            ]
        ];
    }

    public function config($field, $folder, $route)
    {
        $this->setConfig('field', $field);
        $this->setConfig('folder', $folder);
        $this->setConfig('route', $route);
    }

    public function dimensions($dimensions)
    {
        $this->images['dimensions']['default'] = $dimensions;

        if ($this->getImages('thumb')) {
            $this->images['dimensions']['thumb'] = [intval($dimensions[0] / $this->getImages('fallback.percentage')), intval($dimensions[1] / $this->getImages('fallback.percentage'))];
        }

        if ($this->getImages('fallback.active')) {
            $this->images['dimensions']['max'] = [intval($dimensions[0] * ($this->getImages('fallback.percentage')+2)), intval($dimensions[1] * ($this->getImages('fallback.percentage')+2))];
            $this->images['dimensions']['min'] = [intval($dimensions[0] / $this->getImages('fallback.percentage')), intval($dimensions[1] / $this->getImages('fallback.percentage'))];
        }
    }

    public function exact($dimensions)
    {
        $this->images['exact'] = true;
        $this->images['thumb'] = false;

        $this->images['fallback']['active'] = false;

        $this->images['dimensions']['default'] = $dimensions;
        $this->images['dimensions']['thumb'] = $dimensions;
        $this->images['dimensions']['max'] = $dimensions;
        $this->images['dimensions']['min'] = $dimensions;
    }

    public function all($type = 'all')
    {
        $this->images['exact'] = false;
        $this->images['thumb'] = false;
        $this->images['crop'] = false;

        $this->setConfig('type', $type);
    }

    public function load($required = false)
    {
        $this->setConfig('required', $required);

        $this->setConfigApache();

        $this->preventType();

        $this->loadFiles();

        if (is_array($this->files)) {
            $this->checkSizeFiles();
            $this->preventFolders();

            foreach ($this->files as $key => $file) {
                if ($file['error'] == 0 && $file['doop']['valid']) {
                    $this->output[$key] = $this->loadParams($file);

                    if ($this->getConfig('accept') == 'images') {
                        $pathNewFile = $this->getConfig('folder').'/temp/'.$this->output[$key]['name'];
                    }
                    else {
                        $pathNewFile = $this->getConfig('folder').'/'.$this->output[$key]['name'];
                    }

                    if ($this->process($file, $pathNewFile, $this->output[$key])) {
                        if ($this->getConfig('accept') == 'images') {
                            $this->createImages($file, $pathNewFile, $this->output[$key]);
                        }
                    }
                }
            }
        }
    }

    public function loadBlob($blob, $required = false)
    {
        $this->setConfig('required', $required);

        $this->setConfigApache();

        $this->preventType();

        $this->preventFolders(true);

        if ($blob) {
            $this->loadFilesBlob($blob);
        }
    }

    public function loadArchive($file, $fileThumb = null, $required = false)
    {
        $this->setConfig('required', $required);

        $this->setConfigApache();

        $this->preventType();

        $this->preventFolders(true);

        if (!is_null($fileThumb)) {
            if (file_exists($file) && !is_dir($file) && file_exists($fileThumb) && !is_dir($fileThumb)) {
                $this->loadFilesArchive($file, $fileThumb);
            }
        }
        else if (file_exists($file) && !is_dir($file)) {
            $this->loadFilesArchive($file, $fileThumb);
        }
    }

    public function process($file, $pathNewFile, $params)
    {
        if (move_uploaded_file($file['tmp_name'], $pathNewFile)) {
            if (file_exists($pathNewFile)) {
                return true;
            }
        }
    }

    public function output($format = 'json')
    {
        if (!is_null($this->output)) {
            if (!$this->getConfig('multiple')) {
                $this->output = $this->output[0];
            }

            if ($format == 'json') {
                return json_encode($this->output);
            }
            else if ($format == 'array') {
                return $this->output;
            }
        }
        else {
            return null;
        }
    }

    public function loadParams($file)
    {
        $params['name'] = md5(uniqid()).'.'.$file['doop']['ext'];
        $params['realname'] = $file['name'];
        $params['extension'] = strtolower($file['doop']['ext']);
        $params['realname'] = Treatment::generateSlug(str_replace('.'.$params['extension'], '', $file['name']));
        $params['mimetype'] = $file['type'];

        if ($this->getConfig('folderDivider')) {
            $params['folder'] = $this->getFolderType();
        }

        return $params;
    }

    public function setConfigApache()
    {
        ini_set('memory_limit', '100M');
        ini_set('upload_max_filesize', '40M');
        ini_set('post_max_size', '40M');
        ini_set('max_input_time', 3600);
        ini_set('max_execution_time', 3600);
    }

    public function preventType()
    {
        switch ($this->getConfig('type')) {
            case 'png':
            case 'jpg':
            case 'gif':
            case 'webp':
            case 'photo':
                $this->setConfig('accept', 'images');
            break;

            case 'audio':
                $this->setConfig('accept', 'audios');

                $this->resetDimensions();
            break;

            default:
                $this->setConfig('accept', 'files');

                $this->resetDimensions();
            break;
        }
    }

    public function resetDimensions()
    {
        foreach ($this->images['dimensions'] as $key => $value) {
            $this->images['dimensions'][$key] = [0, 0];
        }
    }

    public function loadFiles()
    {
        if ($_FILES) {
            if (array_key_exists($this->getConfig('field'), $_FILES)) {
                $files = $_FILES[$this->getConfig('field')];

                if (!is_array($files['tmp_name']) && $files['tmp_name']) {
                    if ($this->checkErrorUpload($files['error'])) {
                        $this->files[0] = $files;

                        if ($this->files[0]['tmp_name'] && $this->checkAcceptFiles(0)) {
                            $this->setControls('itens', 1);
                            $this->setControls('size', $this->files[0]['size']);

                            $this->files[0]['doop']['ext'] = pathinfo($this->files[0]['name'], PATHINFO_EXTENSION);

                            $this->checkDoopValid(0);
                        }
                    }
                }
                else {
                    foreach ($files as $label => $data) {
                        if ($data && is_array($data)) {
                            foreach ($data as $key => $info) {
                                $this->files[$key]['name'] = $files['name'][$key];
                                $this->files[$key]['type'] = $files['type'][$key];
                                $this->files[$key]['tmp_name'] = $files['tmp_name'][$key];
                                $this->files[$key]['error'] = $files['error'][$key];
                                $this->files[$key]['size'] = $files['size'][$key];
                            }
                        }
                    }

                    if ($this->files && is_array($this->files)) {
                        foreach ($this->files as $key => $value) {
                            if ($this->files[$key]['tmp_name'] && $this->checkAcceptFiles($key)) {
                                $this->setControls('itens', 1);
                                $this->setControls('size', $this->files[$key]['size']);

                                $this->files[$key]['doop']['ext'] = pathinfo($this->files[$key]['name'], PATHINFO_EXTENSION);

                                $this->checkDoopValid($key);
                            }
                        }
                    }
                }
            }
        }
    }

    public function loadFilesBlob($blob)
    {
        $img = \WideImage\WideImage::load($blob);

        if (isset($this->config['type']) && $this->config['type'] != 'photo') {
            $extension = $this->config['type'];
        }

        if ($this->config['type'] == 'photo' || $this->config['type'] == 'video' || $this->config['type'] == 'annex' ) {
            $mimetype = explode(';base64', $blob);
            $mimetype = str_replace('data:', '', $mimetype[0]);

            $checkTypeFiles = $this->typeFormatCheck[$this->config['type']];

            $extension = $this->validMimetypeReturn($checkTypeFiles, $mimetype);
        }

        if ($this->config['name'] == 'automatic') {
            $name = md5(uniqid());
            $filename = $name.'.'.$extension;
        }
        else {
            $name = md5(strtolower($this->config['name']));
            $filename = $name.'.'.$extension;
        }

        $fileTemp = $this->getConfig('folder').'/temp'.'/'.$filename;

        $fileThumb = $this->getFolderPath('thumb', $filename);
        $fileOriginal = $this->getFolderPath('default', $filename);

        $img->saveToFile($fileTemp);
        $img->resize($this->images['dimensions']['default'][0]/2, $this->images['dimensions']['default'][1]/2)->saveToFile($fileThumb);
        $img->resize($this->images['dimensions']['default'][0], $this->images['dimensions']['default'][1])->saveToFile($fileOriginal);

        $realname = explode('.', $filename);

        if ($this->getImages('webpconvert')) {
            $this->convertWeb($fileThumb, $filename, $extension);
            $this->convertWeb($fileOriginal, $filename, $extension);
        }

        $this->output[0]['name'] = $filename;
        $this->output[0]['realname'] = $realname[0];
        $this->output[0]['extension'] = $realname[1];
        $this->output[0]['mimetype'] = mime_content_type($fileOriginal);
        $this->output[0]['folder'] = $this->getFolderType();
    }

    public function loadFilesArchive($file, $fileThumb = null)
    {
        try {
            $img = \WideImage\WideImage::load($file);

            if (isset($fileThumb)) {
                $imgThumb = \WideImage\WideImage::load($fileThumb);
            }
            else {
                $imgThumb = $img;
                $fileThumb = $file;
            }

            $name = md5(uniqid());

            $filename = $name.'.'.pathinfo($file, PATHINFO_EXTENSION);
            $filenameThumb = $name.'.'.pathinfo($fileThumb, PATHINFO_EXTENSION);

            $fileTemp = $this->getConfig('folder').'/temp/'.$filename;

            $fileThumb = $this->getFolderPath('thumb', $filenameThumb);
            $fileOriginal = $this->getFolderPath('default', $filename);

            $img->saveToFile($fileTemp);
            $img->saveToFile($fileOriginal);

            $imgThumb->saveToFile($fileThumb);

            $realname = explode('.', $filename);

            if ($this->getImages('webpconvert')) {
                $this->convertWeb($fileThumb, $filename, $realname[1]);
                $this->convertWeb($fileOriginal, $filename, $realname[1]);
            }

            $this->output[0]['name'] = $filename;
            $this->output[0]['realname'] = $realname[0];
            $this->output[0]['extension'] = $realname[1];
            $this->output[0]['mimetype'] = mime_content_type($fileOriginal);
            $this->output[0]['folder'] = $this->getFolderType();
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getFolderType()
    {
        $params = false;

        if ($this->getConfig('folderDivider')) {
            $params = [];

            if ($this->getConfig('folderDividerYear')) {
                $params['y'] = date('Y');

                if ($this->getConfig('folderDividerMonth')) {
                    $params['m'] = date('m');

                    if ($this->getConfig('folderDividerDay')) {
                        $params['d'] = date('d');
                    }
                }
            }
        }

        return $params;
    }

    public function getFolderPath($subfolder = null, $filename = null)
    {
        if (!is_null($subfolder)) {
            $path = $this->getConfig('folder').'/'.$subfolder;
        }
        else {
            $path = $this->getConfig('folder');
        }

        if ($this->getConfig('folderDivider')) {
            if ($this->getConfig('folderDividerYear')) {
                $path = $path.'/'.date('Y');

                if ($this->getConfig('folderDividerMonth')) {
                    $path = $path.'/'.date('m');

                    if ($this->getConfig('folderDividerDay')) {
                        $path = $path.'/'.date('d');
                    }
                }
            }
        }

        if ($filename) {
            $path = $path.'/'.$filename;
        }

        return $path;
    }

    public function checkDoopValid($position)
    {
        if (array_key_exists($position, $this->files)) {
            if ($this->files[$position]['error'] != 0) {
                $this->files[$position]['doop']['error'] = '501';
            }

            if ($this->getConfig('accept') == 'images') {
                list($width, $height) = getimagesize($this->files[$position]['tmp_name']);

                $this->files[$position]['doop']['width'] = $width;
                $this->files[$position]['doop']['height'] = $height;
            }

            $this->files[$position]['doop']['valid'] = true;
        }
    }

    public function checkErrorUpload($errorNumber)
    {
        switch ($errorNumber) {
            case 0:
                return true;
            break;

            case 1:
                $this->return('upload-size-allowed');
            break;

            case 2:
                $this->return('upload-size-allowed');
            break;

            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
                if ($this->getConfig('required')) {
                    $this->return('upload-required');
                }
                else {
                    return true;
                }
            break;
        }
    }

    public function checkAcceptFiles($position)
    {
        if (isset($this->files[$position]) && $this->files[$position]['type']) {
            if (isset($this->typeFormatCheck[$this->getConfig('type')])) {
                $checkTypeFiles = $this->typeFormatCheck[$this->getConfig('type')];

                if ($this->validMimetype($checkTypeFiles, $this->files[$position]['type'])) {
                    return true;
                }
            }
            else {
                if ($this->validAccept($this->getConfig('type'), $this->files[$position]['type'])) {
                    return true;
                }
            }

            $this->return('upload-invalid-extension');
        }
    }

    public function validMimetype($checkTypeFiles, $type)
    {
        foreach ($checkTypeFiles as $typeFile) {
            if ($this->validAccept($typeFile, $type)) {
                return true;
            }
        }
    }

    public function validMimetypeReturn($checkTypeFiles, $type)
    {
        foreach ($checkTypeFiles as $typeFile) {
            if ($extension = $this->validAcceptReturn($typeFile, $type)) {
                return $extension;
            }
        }
    }

    public function validAccept($typeAccept, $type)
    {
        if (array_key_exists($typeAccept, $this->mimetypesAcceptFiles)) {
            if (is_array($this->mimetypesAcceptFiles[$typeAccept])) {
                foreach ($this->mimetypesAcceptFiles[$typeAccept] as $mimetype) {
                    if ($mimetype == $type) {
                        return true;
                    }
                }
            }
        }
    }

    public function validAcceptReturn($typeAccept, $type)
    {
        if (array_key_exists($typeAccept, $this->mimetypesAcceptFiles)) {
            if (is_array($this->mimetypesAcceptFiles[$typeAccept])) {
                foreach ($this->mimetypesAcceptFiles[$typeAccept] as $mimetype) {
                    if ($mimetype == $type) {
                        return $typeAccept;
                    }
                }
            }
        }
    }

    public function deleteFileMemory($position)
    {
        if (array_key_exists($position, $this->files)) {
            unset($this->files[$position]);

            if (count($this->files) == 0) {
                $this->files = null;
            }
        }
    }

    public function checkSizeFiles()
    {
        if ($this->getControls('size') <= $this->getConfig('limits.size')) {
            return true;
        }
        else {
            $this->return('upload-size-allowed');
        }
    }

    public function preventFolders($force = false)
    {
        if ($this->files || $force) {
            $folder = config('app.ROOT_UPLOAD');

            if (!is_dir($folder)) {
                mkdir($folder, 0777);
            }

            $folder = config('app.ROOT_UPLOAD').'/'.$this->getConfig('folder');

            if (!is_dir($folder)) {
                mkdir($folder, 0777);
            }

            $folder = $folder.'/'.$this->getConfig('accept');

            if (!is_dir($folder)) {
                mkdir($folder, 0777);
            }

            if ($this->getConfig('accept') == 'images') {
                if (!is_dir($folder.'/temp')) {
                    mkdir($folder.'/temp', 0777);
                }

                if (!$this->getConfig('multiple')) {
                    if (!is_dir($folder.'/default')) {
                        mkdir($folder.'/default', 0777);
                    }

                    if (!is_dir($folder.'/thumb')) {
                        mkdir($folder.'/thumb', 0777);
                    }
                }

                if ($this->getConfig('multiple')) {
                    if (!is_dir($folder.'/larger')) {
                        mkdir($folder.'/larger', 0777);
                    }

                    if (!is_dir($folder.'/smaller')) {
                        mkdir($folder.'/smaller', 0777);
                    }
                }
            }

            $this->setConfig('folder', $folder);
        }
    }

    public function createImages($file, $pathFile, $params)
    {
        $img = \WideImage\WideImage::load($pathFile);

        $compression = $this->getImages('compression');

        if ($params['mimetype'] == 'image/png') {
            if ($this->getImages('compression') == 100) {
                $compression = 0;
            }
            else {
                $compression = $this->getImages('compression') / 10;
            }
        }

        $dimensions = $this->images['dimensions'];

        if ($this->getImages('crop') && $this->getConfig('multiple')) {
            $pathSave = $this->getConfig('folder').'/smaller';

            if ($this->checkDimensions($file, $dimensions)) {
                $this->imageSaveResize($img, $pathSave, $compression, $params, $dimensions['thumb']);
            }

            $pathSave = $this->getConfig('folder').'/larger';

            if ($this->checkDimensions($file, $dimensions)) {
                $this->imageSaveResize($img, $pathSave, $compression, $params, $dimensions['default']);
            }
        }

        if ($this->getImages('crop') && $this->getImages('exact')) {
            if ($dimensions['default'][0] == $file['doop']['width'] && $dimensions['default'][1] == $file['doop']['height']) {
                $pathSave = $this->getConfig('folder').'/default';

                $this->imageSave($img, $pathSave, $compression, $params);
            }
            else {
                $this->return('upload-width-height-allowed');
            }
        }

        if ($this->getImages('crop') && $this->getImages('thumb') && !$this->getConfig('multiple')) {
            $pathSave = $this->getConfig('folder').'/thumb';

            if ($this->checkDimensions($file, $dimensions)) {
                $this->imageSaveResize($img, $pathSave, $compression, $params, $dimensions['thumb']);
            }

            $pathSave = $this->getConfig('folder').'/default';

            if ($this->checkDimensions($file, $dimensions)) {
                $this->imageSaveResize($img, $pathSave, $compression, $params, $dimensions['default']);
            }
        }

        if (!$this->getImages('crop') && !$this->getImages('thumb') && !$this->getImages('exact')) {
            $dimensions['default'] = [$file['doop']['width'], $file['doop']['height']];
            $dimensions['thumb'] = [$file['doop']['width']/2, $file['doop']['height']/2];

            $pathSave = $this->getConfig('folder').'/default';
            $this->imageSave($img, $pathSave, $compression, $params);

            $pathSave = $this->getConfig('folder').'/thumb';
            $this->imageSaveResize($img, $pathSave, $compression, $params, $dimensions['thumb']);
        }
    }

    public function checkDimensions($file, $dimensions)
    {
        if ($dimensions['max'][0] >= $file['doop']['width'] && $dimensions['max'][1] >= $file['doop']['height'] || $dimensions['min'][0] <= $file['doop']['width'] && $dimensions['min'][1] <= $file['doop']['height']) {
            return true;
        }
        else {
            $this->return('upload-width-height-allowed');
        }
    }

    public function imageSave($img, $pathSave, $compression, $params)
    {
        $fileSave = $pathSave.'/'.$params['name'];

        $img->saveToFile($fileSave, $compression);

        if ($this->getImages('webpconvert')) {
            $this->convertWeb($fileSave, $params['name'], $params['extension']);
        }
    }

    public function imageSaveResize($img, $pathSave, $compression, $params, $dimension)
    {
        $fileSave = $pathSave.'/'.$params['name'];

        $img->resize($dimension[0], $dimension[1], 'outside')
            ->crop('center', 'center', $dimension[0], $dimension[1])
            ->saveToFile($fileSave, $compression);

        if ($this->getImages('webpconvert')) {
            $this->convertWeb($fileSave, $params['name'], $params['extension']);
        }
    }

    public function convertWeb($originalPath, $originalFileName, $extension)
    {
        if ($originalPath && $originalFileName && $this->getImages('webpconvert')) {
            if (file_exists($originalPath)) {
                $ext = strtolower(pathinfo($originalFileName, PATHINFO_EXTENSION));
                $dirPaths = str_replace($originalFileName, '', $originalPath);
                $filename = str_replace('.'.$ext, '', $originalFileName);
                $imgWebp = $dirPaths . $filename . '.webp';

                switch ($ext) {
                    case 'jpg':
                    case 'jpeg':
                        $gdImageInstance = imagecreatefromjpeg($originalPath);

                        $conversionSuccess = imagewebp($gdImageInstance, $imgWebp, 75);
                    break;

                    case 'gif':
                        $gdImageInstance = imagecreatefromgif($originalPath);
                                             imagepalettetotruecolor($gdImageInstance);
                        imagealphablending($gdImageInstance, true);
                        imagesavealpha($gdImageInstance, true);

                        $conversionSuccess = imagewebp($gdImageInstance, $imgWebp, 75);
                    break;

                    case 'png':
                        $gdImageInstance = imagecreatefrompng($originalPath);

                        imagepalettetotruecolor($gdImageInstance);
                        imagealphablending($gdImageInstance, true);
                        imagesavealpha($gdImageInstance, true);

                        $conversionSuccess = imagewebp($gdImageInstance, $imgWebp, 85);
                    break;
                }
            }
        }
    }

    public function return($code)
    {
        if (is_null($this->callback)) {
            header('Location: '.$this->getConfig('route').'/&callback='.$code);
            exit;
        }
        else {
            $this->callback->set($code, $this->getConfig('route'));

            exit;
        }
    }
}
